unit DataConnection;
{
    UNIT DataConnection;
    Version number 1.0(beta)

This unit contains the data connection manager TDataConnection class.
All methods are described in the interface part.

Notes:
    * currently no support built-in for passive data transfers, only active
      transfer method supported.
    * TDataConnection must re-instanted on every data connection, for example
      after an AbortOperation() or after a successfull data transfer.

    * v1.01 24th of August, '99: Bug fixed in aborting (ReceiveFromServer)

Created by Pter Karsai, June '99.

}

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, WSocket, Forms, ExtCtrls,
     Winsock;

{ constans }
const otSending     = 0;   { otX constans are determines the current }
      otReceiving   = 1;   { operation's type. }
      otNoOperation = 2;

      TransferBufferLength = 1514;  { internal transfer buffer length }
      IOBufferLength       = TransferBufferLength * 16; { currently 24224 }
      { Internal storage of received/send data. Increasing this value will cause
        fewer disk access, but increasing data lost if a system crash occur.
        IOBufferLength must be divideable by TransferBufferLength. }

{ Note: TransferBuffer must be shorter than IOBuffer. IOBuffer wouldn't be
  higher than 32767. }

type TOutDial = procedure of object;  { type TOutDial simply call-type for
                                        external parameter-given procedures }

type
  ETSocketException    = class(Exception);
  { on any socket I/O error }
  ETNotConnected       = class(Exception);
  { on I/O operations detected on not connected sockets }
  ETConnectingTimedOut = class(Exception);
  { on connecting has timed out }


  TDataConnection = class(TObject)
  private
  { internal data }
{ component control and socket data }
    Owner            : TComponent;  { owner of this object }
    LocalSrvSocket   : TWSocket; { listening socket, used by d/l }
    LocalCliSocket   : TWSocket; { client socket, used by d/l and u/l both }
{ timing }
    TimeOutTimer     : TTimer;   { internal timer to check timeouts }
    TimeSpent        : word;     { TimeOut seconds counter }
{ synchronization }
    CliConnected     : boolean;  { TRUE if client connected }
    DataSent         : boolean;  { TRUE current data sent }
    DoAbort          : boolean;  { if TRUE, send/receive will stop ASAP }
    OperationType    : byte;     { current operation's type }
    IOErrorMsg       : string;   { saved EInOutError.Message, it's required to
                                   re-raise EInOutError if that raised in an
                                   event. }
{ data saving }
    DataFileName     : string;   { filename of the current operation }
    DataFile         : file;     { file handle of the current operation }
    TransferBuffer   : array[0..TransferBufferLength - 1] of byte;
    IOBuffer         : array[0..IOBufferLength - 1] of byte;
    IOBufferTop      : integer;  { index of the last byte in IOBuffer }
    TransferedBytes  : longint;  { transfered bytes number of the *current*
                                   operation }
    TotalTransfered  : longint;  { transfered bytes + filesize or offset }
{ ----------------------------------------------------------------------------}
{ private service methods }
{ ----------------------------------------------------------------------------}
   procedure StartTimer(TimeOut: word);
{ StartTimer starts the TimeOut timer. }

   procedure StopTimer;
{ StopTimer stops the TimeOut timer. }

   function HasTimedOut(TimeOut: word): boolean;
{ Returns TRUE, if current operation timed out, otherwise FALSE }

   procedure SendIOBuffer;
{ Function: 'flush' IOBuffer to remote host. This is a blocking method, it
  doesn't return until the buffer flushed. }

   function PrepareDataConnection: word;
{ Function: prepare local listening socket, bind event handlers.
  Return with port number of listening socket. Method switches local server
  socket to listening state. }

   procedure ShutDownIO;
{ Function: shut down all I/O TDataConnection use. If sockets connected,
  they'll be disconnected first. }

   procedure GetServerConnection(TimeOut: word);
{ Waits till data connection built. TimeOut check as elsewhere in the class. }

  public

  OnDataConClosed    : TOutDial; { added 7th of August, '99 }
  OnDataConConnected : TOutDial; { it's called when data connection connected.
                                   Added 8th of August, '99 }
{ ----------------------------------------------------------------------------}
{ data connection management }
{ ----------------------------------------------------------------------------}

   function PrepareSending(onCliConnected: TOutDial): word;
{ Method PrepareSending() prepares LocalSrvSocket to receive server's
  connection request. It returns the port number of the dynamically allocated
  server socket. You should use this method before SendToServer(). If you
  don't know why, I suppose you haven't read about FTP. RFTM, lame :)

  'onCliConnected' procedure will called if connection estabilished. If you
  give 'nil' reference as parameter, no method will called.

  Throws ETSocketException on any socket error. }

   function PrepareReceiving(AppendToFile: boolean; onCliConnected: TOutDial;
                             var FSize: longint): word;
{ Method PrepareReceiving() prepares datafile and LocalSrvSocket. Returns
  with port number of the listening socket required by PORT command. Open
  the datafile, too.
  You should call this method before calling ReceiveFromServer().

  'onCliConnected' procedure will called if connection estabilished. If you
  give 'nil' reference as parameter, no method will called.

  Throws ETSocketException on any socket error.
  Throws EInOutError on any disk I/O error. }

    procedure ReceiveFromServer(TimeOut: word);
{ Receive data from server and save to file given in the constructor. It is not
  the real receiver part, it just provide blocking control while receiving, it
  means, ReceiveFromServer() will return when receiving finished. It's not sure
  you don't receive data before call this method - receiving is automatical,
  but you have to call this method, because it'll free allocated system
  resources such as sockets and datafile.

  This is a *blocking* method, it does *not* return until the file transferred
  or an unforeseen error detected.

  Throws EInOutError on any file I/O error,
  Throws ETSocketException on any socket I/O error.
  Throws ETConnectingTimedOut on timeout }

    procedure SendToServer(SendFromOffset: longint; TimeOut: word);
{ Sends file given in the constructor. File sending starts at SendFromOffset,
  if 0 specified, total file will be sent.

  You can define a timeout limit by parameter 'TimeOut'. If TimeOut 0, no
  timeout check defined, else SendToServer() will wait max. TimeOut seconds
  before return, if no connection built.

  This is a *blocking* method, it does *not* return until the file transferred
  or an unforeseen error detected.

  This is the standard sending method ('active transfer'), you should use it
  if you didn't send a PASV command before.

  You *MUST* call PrepareSending() method before call this method; you
  gotta send a PORT command with some port info you can get by PrepareSending().
  If you don't know what I'm talkin' about, RFTM :)

  Throws EInOutError on any file I/O error,
  Throws ETNotConnected on server disconnect during the send operation,
  Throws ETSocketException on any socket I/O error.
  Throws ETConnectingTimedOut on timeout }

    procedure GetTransferInfo(var TotalTransferedP, TransferedBytesP: longint);
{ GetTransferInfo() returns with information about the current transfer.
  'TotalTransfered' is the total number of bytes got/sent, 'Transfered' is
  the number of bytes got/sent by *current* operation. }

    procedure AbortOperation;
{ By calling AbortOperation() method you can abort the current operation
  [ReceiveFromServer() and SendToServer()]. }

    function CurrentOperation: byte;
{ Return with the current operation type as otX* constants }
{ ----------------------------------------------------------------------------}
{ event handlers }
{ ----------------------------------------------------------------------------}
{ client events }
    procedure CliDataSent(Sender: TObject; Error: Word);
    procedure CliDataAvailable(Sender: TObject; Error: Word);
    procedure CliSessionClosed(Sender: TObject; Error: Word);

{ server event }
    procedure LocalSrvSessionAvailable(Sender: TObject; Error: Word);

{ TimeOutTimerTick is an OnTimer event of TimeOutTimer }
    procedure TimeOutTimerTick(Sender: TObject);

{ ----------------------------------------------------------------------------}
{ constructor and destructor }
{ ----------------------------------------------------------------------------}
    constructor Create(xFileName: string; AOwner: TComponent);
    destructor Destroy; override;
    procedure Free;
end;

implementation
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ private services ------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TDataConnection.StartTimer(TimeOut: word);
begin
{ prepare TimeOutTimer timer and start it }
     TimeSpent:= 0;
     if TimeOut > 0 then
        TimeOutTimer.Interval:= 1000 { Interval given in milliseconds }
     else
        TimeOutTimer.Interval:= 0; { on interval 0, no timer called }
{ start timer }
     TimeOutTimer.Enabled:= true;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.StopTimer;
begin
     TimeOutTimer.Enabled:= false;
end;

{------------------------------------------------------------------------------}

function TDataConnection.HasTimedOut(TimeOut: word): boolean;
begin
{ return TRUE if timer started... }
     Result:= TimeOutTimer.Enabled and (TimeOutTimer.Interval > 0);
{ if timer started and time spent not reach limit, return FALSE }
     if Result then
         if TimeSpent <= TimeOut then Result:= false;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.SendIOBuffer;
var cIOBufferPos: word;    { current IOBuffer position }
    bytesSent   : integer; { bytes sent in one round }
    toSend      : word;    { bytes to send }
    shouldSend  : boolean; { cycle control }
begin
{ start send at 0 }
     cIOBufferPos:= 0; toSend:= TransferBufferLength;
     shouldSend:= true;
{ send data while we should send }
     while shouldSend do begin
           shouldSend:= true;
     { if this pack of data is the last in the IOBuffer... }
           if cIOBufferPos + TransferBufferLength = IOBufferLength then
              shouldSend:= false;
     { if there are enough data to send a full TransferBuffer...}
           if cIOBufferPos + TransferBufferLength < IOBufferTop then
              toSend:= TransferBufferLength
           else
          { if there are no more data (IOBuffer is not full with data )}
              if IOBufferTop < IOBufferLength - 1 then
              begin
                 toSend:= IOBufferTop - cIOBufferPos + 1;
                 shouldSend:= false;
              end;

     { synchronize }
           DataSent:= false;
     { send data thru - ignore exceptions }
           bytesSent:= LocalCliSocket.Send(@(IOBuffer[cIOBufferPos]), toSend);

     { wait 'till data sent }
           while not DataSent and CliConnected and ((WSAGetLastError = 0) or
                (WSAGetLastError = WSAEWOULDBLOCK)) and not DoAbort do
           { Winsock error code 10035: WSAEWOULDBLOCK. It's normally received
             when using asychronous, non-blocking sockets. ~It's Winsock's way
             of telling your program "I can't do that right now, because I would
             have to block to do so"~ }
           begin
                 Application.ProcessMessages;
                 if Application.Terminated then Exit;
           end;

     { check correct of data sent }
           if not CliConnected and not DoAbort then
              raise ETNotConnected.Create('Server dropped data connection, ' +
                                          'file send failed.');

           if (WSAGetLastError <> 0) and (WSAGetLastError <> WSAEWOULDBLOCK)
           then raise ETSocketException.CreateFmt('Data socket error %d: %s.',
              [WSAGetLastError, WSocketErrorDesc(WSAGetLastError)]);

           cIOBufferPos:= cIOBufferPos + toSend;

     { set transfer data }
           TransferedBytes:= TransferedBytes + bytesSent;
           TotalTransfered:= TotalTransfered + bytesSent;
     end;
end;

{------------------------------------------------------------------------------}

function TDataConnection.PrepareDataConnection: word;
begin
{ set up server socket }
     LocalSrvSocket:= TWSocket.Create(Owner);
     LocalSrvSocket.Addr:= '0.0.0.0';
     { we accept connections from any addresses, because some FTP services
       are on different IP's - FTP service running on more than one machine. }
     LocalSrvSocket.Port:= '0';  { we don't specify port, system will give one }
     LocalSrvSocket.Proto:= 'tcp';  { what's this TCP shit? :) }

{ bind event handlers }
     LocalSrvSocket.OnSessionAvailable:= LocalSrvSessionAvailable;

{ try to switch server to listening state }
     try
        LocalSrvSocket.Listen;
     except
        on Exception do
           raise ETSocketException.CreateFmt('Data socket error %d: %s',
          [WSAGetLastError, WSocketErrorDesc(WSAGetLastError)]);
     end;

{ wait while switching }
     while not (LocalSrvSocket.State in [wsListening]) do begin
           Application.ProcessMessages;
           if Application.Terminated then exit;
     end;
{ return port }
     Result:= StrToInt(LocalSrvSocket.GetXPort);

{ examine errors }
     if (WSAGetLastError <> 0) and (WSAGetLastError <> WSAEWOULDBLOCK) then
        raise ETSocketException.CreateFmt('Data socket error %d: %s',
              [WSAGetLastError, WSocketErrorDesc(WSAGetLastError)]);
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.ShutDownIO;
begin
{ shut down timer first }
     if Assigned(TimeOutTimer) then begin
         if TimeOutTimer.Enabled then TimeOutTimer.Enabled:= false;
         TimeOutTimer.Free;
         TimeOutTimer:= nil;
     end;

{ shut down LocalCliSocket first }
     if Assigned(LocalCliSocket) then begin
         if CliConnected then LocalCliSocket.Close;
         LocalCliSocket.Free;
         LocalCliSocket:= nil;
     end;

{ shut down LocalSrvSocket }
     if Assigned(LocalSrvSocket) then begin
         if LocalSrvSocket.State = wsListening then LocalSrvSocket.Close;
         LocalSrvSocket.Free;
         LocalSrvSocket:= nil;
     end;

{ shut down file }
     try
         CloseFile(DataFile);
     except
         on Exception do;
     end;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.GetServerConnection(TimeOut: word);
begin
{ wait for server connection }
     StartTimer(TimeOut);
     while not CliConnected and not HasTimedOut(TimeOut) and
           (OperationType <> otNoOperation) do
     begin
           Application.ProcessMessages;
           if Application.Terminated then
           begin
                StopTimer;
                exit;
           end;
     end;
     StopTimer;

{ perform timeout-check }
     if HasTimedOut(TimeOut) and not CliConnected then
     begin
        { shut down I/O prepares }
        ShutDownIO;
        raise ETConnectingTimedOut.Create('Server connection request has ' +
                                          'timed out.');
     end;

{ maybe list already got... on fast connections }
     if not CliConnected and (OperationType <> otNoOperation) then begin
       ShutDownIO;
       raise ETNotConnected.Create('Remote host didn''t request connection.')
     end;
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ data connection management --------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
function TDataConnection.PrepareSending(onCliConnected: TOutDial): word;
begin
{ setup listening socket }
     Result:= PrepareDataConnection;
{ set operation type }
     OperationType:= otSending;
{ set "event handler" added 8th of August, '99 }
     OnDataConConnected:= OnCliConnected;
end;

{------------------------------------------------------------------------------}

function TDataConnection.PrepareReceiving(AppendToFile: boolean;
                            OnCliConnected: TOutDial; var FSize: longint): word;
begin
     AssignFile(DataFile, DataFileName);
{ no i/o check 'cause all i/o error passed up to the higher level }
     if AppendToFile then
        reset(DataFile, 1)     { if append, just reset file... }
     else
        rewrite(DataFile, 1);  { ...anyway create it }
     seek(DataFile, FileSize(DataFile));   { seek EOF }
     FSize:= FileSize(DataFile);

{ save number of bytes we've already got }
     TotalTransfered:= filesize(DataFile);

{ setup listening socket }
     Result:= PrepareDataConnection;

{ set operation type }
     OperationType:= otReceiving;

{ set "event handler" added 8th of August, '99 }
     OnDataConConnected:= OnCliConnected;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.ReceiveFromServer(TimeOut: word);
begin
{ wait for server's connection request }
     GetServerConnection(TimeOut);

{ set operation type }
     OperationType:= otReceiving;

{ wait until the operation finished/error occured/cancelled }
     while CliConnected and ((WSAGetLastError = 0) or
           (WSAGetLastError = WSAEWOULDBLOCK)) and not DoAbort and
           (IOErrorMsg = '') do
     begin
           Application.ProcessMessages;
           if Application.Terminated or DoAbort then
           begin
              ShutDownIO;
              exit;
           end;
     end;

{ set operation type }
     OperationType:= otNoOperation;

     if IOErrorMsg <> '' then raise EInOutError.Create(IOErrorMsg);

{ flush i/o buffer, if not aborting in progress }
     if not DoAbort then
     begin
        blockwrite(DataFile, IOBuffer, IOBufferTop);
        CloseFile(DataFile);
     end;

{ close i/o devices }
     ShutDownIO;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.SendToServer(SendFromOffset: longint; TimeOut: word);
begin
{ setup starting transfer values }
     TotalTransfered:= SendFromOffset;

{ open datafile }
     AssignFile(DataFile, DataFileName);
     { no i/o check 'cause all i/o error passed up to the higher level }
     reset(DataFile, 1);
     seek(DataFile, SendFromOffset);

{ wait for server's connection request }
     GetServerConnection(TimeOut);

     OperationType:= otSending;

{ send the file }
     while not Eof(DataFile) and CliConnected and (WSAGetLastError = 0)
           and not DoAbort do
     begin
           blockread(DataFile, IOBuffer, IOBufferLength, IOBufferTop);
           IOBufferTop:= IOBufferTop - 1;
     { 'flush' IOBuffer to remote host if data remaining }
           try
               if IOBufferTop > 0 then SendIOBuffer;
           except
           { on any exceptions, close DataFile }
               on Exception do
               begin
                  CloseFile(DataFile);
                  raise;  { re-raise exception }
               end;
           end;
     end;

     OperationType:= otNoOperation;

{ if the server drops the connection, close datafile }
     if not CliConnected and not DoAbort then
     begin
          ShutDownIO;
          raise ETSocketException.Create('Server dropped data connection, ' +
                                         'file send failed.');
     end;

{ close i/o devices }
     ShutDownIO;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.GetTransferInfo(var TotalTransferedP,
                                          TransferedBytesP: longint);
begin
     TotalTransferedP:= TotalTransfered;
     TransferedBytesP:= TransferedBytes;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.AbortOperation;
begin
     DoAbort:= true;
end;

{------------------------------------------------------------------------------}

function TDataConnection.CurrentOperation: byte;
begin
     Result:= OperationType;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ event handlers --------------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TDataConnection.CliDataSent(Sender: TObject; Error: Word);
begin
     DataSent:= true;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.CliDataAvailable(Sender: TObject; Error: Word);
var receivedBytes: word;  { number of received bytes }
      trashBuffer: array[0..1024] of byte;
begin
{ we don't wait any data while we sending, but the Devil never sleeps. Trash
  incoming data. }
     if OperationType = otSending then
        receivedBytes:= LocalCliSocket.Receive(@trashBuffer,
                                               sizeof(trashBuffer));

{ if there was no error, receive data }
     if (Error = 0) and (OperationType = otReceiving) then begin
     { if it's possible to buffer overrun, flush buffer }
        if IOBufferTop + TransferBufferLength > IOBufferLength - 1 then
        begin
        { save error if occur while writing }
           try
              blockwrite(DataFile, IOBuffer, IOBufferTop);
           except
              { strange way to save an exception but I don't know better -
                I'm an amateur Delphi programmer :( }
              on E:EInOutError do IOErrorMsg:= E.Message;
           end;
           IOBufferTop:= 0;
        end;

     { receive data }
        receivedBytes:= LocalCliSocket.Receive(@TransferBuffer,
                                               TransferBufferLength);

     { on disconnecting, there're some false onDataAvailable call, receive
       buffer doesn't contain data. If number of received bytes is zero,
       it's a fake call, sure. Noone can bother me :) }
        if receivedBytes <> 0 then begin
           move(TransferBuffer, IOBuffer[IOBufferTop], receivedBytes);
           IOBufferTop:= IOBufferTop + receivedBytes;
        end;

     { save transfer-statistics data }
           TransferedBytes:= TransferedBytes + receivedBytes;
           TotalTransfered:= TotalTransfered + receivedBytes;
     end;

     Application.ProcessMessages;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.CliSessionClosed(Sender: TObject; Error: Word);
begin
     CliConnected:= false;
     OperationType:= otNoOperation;  { added 8th of August, '99 }
     if Assigned(OnDataConClosed) then OnDataConClosed;
     { added 7th of August, '99 }
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.TimeOutTimerTick(Sender: TObject);
begin
{ TimeSpent is word, protect it from overflow }
     if TimeSpent < $ffff then inc(TimeSpent);
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.LocalSrvSessionAvailable(Sender: TObject; Error: Word);
begin
{ protect against multiple connections - TWSocket hasn't got control to limit
  maximal number of clients can connect to server socket. }
    if not CliConnected then begin
    { create socket }
       LocalCliSocket:= TWSocket.Create(Owner);
       LocalCliSocket.LingerOnOff:= wsLingerOff;  { avoid broken pipe }
    { bind event handlers }
       LocalCliSocket.OnDataSent:= CliDataSent;            { for send }
       LocalCliSocket.OnDataAvailable:= CliDataAvailable;  { for receive }
       LocalCliSocket.OnSessionClosed:= CliSessionClosed;  { have a nice day }
    { accept incoming session }
       LocalCliSocket.Dup(LocalSrvSocket.Accept); { ELP rules! I think :) }
    { call external event handler if assigned - added on 8th of August, '99. }
       CliConnected:= true;
       if Assigned(OnDataConConnected) then OnDataConConnected;
    end;
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ constructor/destructor methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
constructor TDataConnection.Create(xFileName: string; AOwner: TComponent);
begin
     inherited Create;
{ save parameters }
     DataFileName:= xFileName;
     Owner:= AOwner;

{ setup starting values }
     IOBufferTop:= 0;
     TransferedBytes:= 0;
     TotalTransfered:= 0;
     DoAbort:= false;
     IOErrorMsg:= '';
     OperationType:= otNoOperation;

{ create timeout timer }
     TimeOutTimer:= TTimer.Create(AOwner);
     TimeOutTimer.Enabled:= false;
     TimeOutTimer.OnTimer:= TimeOutTimerTick;
end;

{------------------------------------------------------------------------------}

destructor TDataConnection.Destroy;
begin
     ShutDownIO;
     if Assigned(TimeOutTimer) then TimeOutTimer.Free;
     inherited Destroy;
end;

{------------------------------------------------------------------------------}

procedure TDataConnection.Free;
begin
     if Self <> nil then Destroy;
end;

end.
